/*
 * Copyright (c) 2004, Freescale Semiconductor
 * Freescale Confidential Proprietary
 *
 */
/*!
 * \file simple_phy.c
 *
 * \brief  This is the SMAC C code for the PHY layer  
 *
 * \author a19259 
 *
 * \version 4.1a
 *
 * \date - 29/07/2005 03:30:28
 *
 * \b Description: This is the SMAC C source physical layer file for the HC(S)08
 *                 MCU and MC13192 transceiver.
 *                 The SMAC phy is the second lowest layer of C code.
 * 
 * \b Project name: SMAC (Simple Media Access Controller)
 *
 * \b Department : Freescale Radio Products Division
 *
 * \b History: 
 * - 16/11/2005 Doc. update to Doxygen compliant by Laura Delgado r12370c
 * 
 */

#include "hprfgw_config.h"
#include "smac_MC13192_regs.h"
#include "smac_MC13192_hw_config.h"
#include "smac_pub_def.h"
#include "smac_drivers.h"
#include "smac_simple_phy.h"
#include "smac_simple_mac.h"
#include "smac_mcu_hw_config.h"
#include "smac_mcu_spi_config.h"
#include "smac_app_config.h"
#include "smac_app.h"
#include "smac_802_15_4_shim.h"
//#include "hprfgw_rfSharedDefs.h"
//#include "hprfgw_rfSlaveIntToHost.h"
//#include "hprfgw_rfSlaveIntToRfApp.h"
//#include "stdlib.h"

/* Globals */
tPhyOptions gsPhyOptions;
int gTxInProgress = 0;
int gTxWaitForCCA = 0;
int gTxLOUnlocked = 0;
// MNT - 10/3/2007 - Track the number of CCA failures
UINT32 Rfdiags_NumCCAFailures = 0;
UINT32 Rfdiags_NumCCATimeoutsFailures = 0;

UINT32 Rfdiags_getNumCCAFailures(void) { return Rfdiags_NumCCAFailures; }
UINT32 Rfdiags_getNumCCATimeoutsFailures(void) { return Rfdiags_NumCCATimeoutsFailures; }

extern UINT16 g_u16StatusContent;
extern UINT32 Smac_numTimesChanBusy;
extern UINT32 Smac_ChanBusyResetFlag;


/* Version string to put in NVM. Note! size limits */
/* Normally it shoud be enough to change the version numbers. */
#define PHY_Version     "4.2 "
#define PHY_Label       "SMAC"
#define PHY_VerComplete "PHY " PHY_Label " Ver " PHY_Version " Build: " __DATE__ " " __TIME__
#define PHY_VerSize     sizeof(PHY_VerComplete)


#if defined (HCS08G) | defined (HCS08R)	| defined (_HCS12)
  #include <hidef.h>
#pragma MESSAGE DISABLE C3303  /* 
                                * Warning C3303: Implicit concatenation of 
                                * strings 
                                */
#pragma MESSAGE DISABLE C4200  /* 
                                * Warning C4200: Other segment than in previous 
                                * declaration
                                */
#pragma CONST_SEG BOOTLOADER_PHY_NV_DATA0
#endif

// DO NOT CHANGE OR REMOVE

// This string will be located in the NV RAM0 section.
// Note!! Check that item is location in the sequence as specified. 
const unsigned char SPHY_Version[PHY_VerSize] = PHY_VerComplete;

#if defined (HCS08G) | defined (HCS08R) | defined (_HCS12)
#pragma CONST_SEG DEFAULT
#endif




/*
 * PLMEPhyInit : Sets the default values for Phy
 *
 * Parameters : Uses the gsPhyOptions
 *
 * Return : Status
 */
 
 UINT8 PLMEPhyReset(void) {
 
   /* Sets the default Options */
   gsPhyOptions.Bits.u1PromiscuousMode = SMAC_PROMISCUOUS_MODE;
   gsPhyOptions.Byte.u16ScanChannels = 0xFFFF; /* Default to all channels */
   
   return SMAC_SUCCESS;  
 }
/*!
 * \fn UINT8 PDDataRequest(tTxPacket *psPacket, UINT8 shouldWaitForCC, UINT8 fnNotCalledInISR) 
 *
 * \brief Transmit data packet
 *
 * \param *psPacket packet pointer
 *
 * \return Status
 */


UINT8 PDDataRequest(tTxPacket *psPacket, UINT8 shouldWaitForCC, UINT8 fnNotCalledInISR)
{
    // MNT - 6/5/2007 - This is a 25 ms timeout. This should eventually
    // be randomized for the CCA
    // MNT - 7/2/2007 - Change the CCA timeout. Each CCA transaction has a timeout of 350 us
    // We will now only try 50 tries to access the channel - Total time to give up on access
    // is 400us x 50 = 20 ms.
    // MNT - 7/5/2007 - This code now implements the backoff algorithm defined in Section 
    // 7.5.1 of the 802.15.4 spec.
    INT32 timeout;
    UINT32 randWaitPeriod;
    UINT32 randWaitMaxPeriod;
    UINT8 numBackOffs = 0;
    UINT8 currBE = Rf_Smac_802_PIB.minBE;

    HPRF_DEBUG(DmConsolePrintf("========== %s ========== psPacket = %p size = %u waitForCC = %u !inISR = %u\n", \
            __FUNCTION__, psPacket, psPacket->u8DataLength, shouldWaitForCC, fnNotCalledInISR);)
    if (gu8RTxMode == IDLE_MODE) {
#if (SMAC_FEATURE_SECURITY == TRUE)
        SECSecure(psPacket->pu8Data, psPacket->u8DataLength);
#endif /* SMAC_FEATURE_SECURITY */

#if SMAC_SM_DEBUG
//        RAMDrvReadTx(SMAC_DEBUG_PACKET_READ, psPacket->u8DataLength);
#endif

        HPRF_PROFILE(PktWrite);
        // Wait till we get a clear channnel. Modify this to take out the while loop

        if (shouldWaitForCC == TRUE) {
            SMAC_LOG_DEBUG_STATE(gi8AppStatus, SMAC_SM_DEBUG_EVENT_TX_STARTCCA);

            // MNT - 7/5/2007 - Instead of just waiting for a timeout, implement the
            // random backoff algorithm based on 7.5.1 of the 802.15.4 spec.
            while (numBackOffs < SMAC_802_PIB_MAXCSMABOFF_DEFAULT) {
                // Step 1 - Get a random delay period between 0 and (2^BE - 1) backoff periods
                // Diab rand goes from 0 to 32727. So, all we need to do is take a modulus
                randWaitMaxPeriod = ((2 << currBE) - 1);
                if (randWaitMaxPeriod) {
                    randWaitPeriod = (rand() % randWaitMaxPeriod) * SMAC_802_PIB_BACKOFF_PERIOD_INUS;
                    HwDelayUsec(randWaitPeriod);
                    //RFIntHostApp.pF_HwDelayUsec(randWaitPeriod);
                } // if (randWaitMaxPeriod)

                // MNT - 7/5/2007 - If channel is busy, increment number of backoffs and
                // the exponent. Otherwise transmit
                if (PLMECCARequest() == 1) {
                    numBackOffs++;
                    if ((currBE + 1) < SMAC_802_PIB_MAXBE) {
                        currBE++;
                    } //if ((currBE+1) < SMAC_802_PIB_MAXBE)
                    else {
                        currBE = SMAC_802_PIB_MAXBE;
                    } //if ((currBE+1) < SMAC_802_PIB_MAXBE)

                    SMAC_LOG_DEBUG_STATE(gi8AppStatus, SMAC_SM_DEBUG_EVENT_TX_CCA_CHBUSY);
                    // MNT - 7/7/07 Increment the channel busy counter. Currently, we retry
                    // 4 times and try to access the channel 4 times. If we were busy for 75 % of
                    // the time, set a flag to reset when we go into receive mode.
                    // SMAC_802_PIB_MAXCSMABOFF_DEFAULT == 4
                    // SMAC_RETRY_COUNT == 4
                    Smac_numTimesChanBusy++;
                    if (Smac_numTimesChanBusy >= (SMAC_802_PIB_MAXCSMABOFF_DEFAULT * SMAC_RETRY_COUNT)) {
                        Smac_ChanBusyResetFlag = 1;
                    }

                } // if (PLMECCARequest() == 1)
                else {
                    // MNT - 7/5/2007 - Channel is idle. Break out of the while loop
                    break;
                } // if (PLMECCARequest() == 1)

            } // while (numBackOffs > SMAC_802_PIB_MAXCSMABOFF_DEFAULT)

            SMAC_LOG_DEBUG_STATE(gi8AppStatus, SMAC_SM_DEBUG_EVENT_TX_ENDCCA);
            // At this point, if CCA failed, just return an error condition
            if (numBackOffs >= SMAC_802_PIB_MAXCSMABOFF_DEFAULT) {
                return SMAC_CCA_FAILED;
            } // end if (numBackOffs >= SMAC_802_PIB_MAXCSMABOFF_DEFAULT)

        } // if (shouldWaitForCC == TRUE)

        HPRF_PROFILE(CCADone);
        // MNT - 6/15/2007 - Flag the initiation of the transmission.
        gTxInProgress = 1;
        gTxLOUnlocked = 0;
        gTxWaitForCCA = shouldWaitForCC;

        // MNT - 10/18/2007 - Clear the Txdone event. 
#if defined (OS_NUCLEUS)
        NU_Set_Events(&TxRxTask.txEvent, ~RF_TX_TXDONE, NU_AND);
#elif defined (OS_WINCE)
        ResetEvent(TxRxTask.hTxEvent);
#endif
        //TODO: Cleanup
        //OsClearEvent(TxRxTask.hTxEvent, RF_TX_TXDONE);
        OsWaitEvent(TxRxTask.hTxEvent, OS_NO_WAIT);
//        pr_info("========= %s ========= Just before write packet\n", __FUNCTION__);

        // MNT - 10/17/2007 - Make sure that you fill the buffer right before
        // you transmit the data. This way if we have to send an ACK, it won't 
        // corrupt the buffer in the PHY chip. Code Review 10.17.07
        RAMDrvWriteTx(psPacket); /* Load the data into packet RAM */
        (void) PLMESetTrxStateRequest(TX_MODE); /* transmit it */

        // MNT - 5/25/2007 - We will never go to idle mode if we are sending
        // an ACK because it is being done in the ISR. So, we should query the
        // chip as if we are in the ISR. We know that the we are transmitting
        // an ACK if we are not waiting for CCA. Just stall till we get an
        // indication from the chip that the TX is done

        // MNT - 6/5/2007 - This is a 25 ms timeout. Reduce it if necessary
        // MNT - 6/15/2007 - We now wait for the mode to go to RXWTO instead of
        // idle because we start waiting for an ACK right away
        // MNT - 6/25/2007 - added a separate condition to indicate if the data request
        // is being done in an ISR
        // MNT - 10/18/2007 - The timeout should be set to 2 because now we are going to
        // rely on an event coming from the ISR instead
#if defined (OS_NUCLEUS)
        timeout = 2;
#elif defined (OS_WINCE)
        timeout = 20;	// Specify in milli seconds		// PRASAD - need constant define?
#elif defined (OS_LINUX)
        timeout = 20;
#endif

        if (fnNotCalledInISR == TRUE) {
            /* Wait for the state to return to idle. (finish transmitting) */
            // MNT - 10/18/2007 - Instead of waiting in a tight loop, now we will wait for an
            // event coming from the ISR.
            // MNT - 10/18/2007 - Wait one tick per iteration. The last parameter is a 1 for one tick
            if (OsWaitEvent(TxRxTask.hTxEvent, timeout)) {
                // We got a valid event.
                return SMAC_SUCCESS;
            } // if retrieve events
            else {
                return SMAC_FAILURE;
            }

        } // else if (fnNotCalledInISR == TRUE)
        else {
#if defined (OS_NUCLEUS)          
            timeout = 0x650;
#elif defined (OS_WINCE)
            // OMAP processor runs faster, so timeout increase??
            timeout = 0x2000;// PRASAD - check with MNT. Initially 0xA000, and was working.
#elif defined (OS_LINUX)
            timeout = 0x2000;
#endif
            /* Wait for the state to return to idle. (finish transmitting) */
            // MNT - 5/25/2007 - To wait for state to become idle, poll the
            // IRQ pin
#if defined (OS_NUCLEUS)
            while ((IRQFLAG() == 0) && (timeout--))
            {
                asm("nop");
            }
#elif defined (OS_WINCE)
            while (IRQFLAG() == 0)
            {
                timeout--;
                if (timeout == 0)
                {
                    timeout = -1;
                    break;
                }
            }
#endif
            while (IRQFLAG() == 0) {
                timeout--;
                if (timeout == 0) {
                    timeout = -1;
                    break;
                }
            }
            // At this point, call the mini MC13192 ISR to handle any extraneous events
            // in addition to the tx done event
            // MNT - 5/29/2007 - Only call this if we have not timed out
            if (timeout != -1) {
                MC13192_MiniISRForACK();
            } // end if no timeout

        } // end if (fnNotCalledInISR == TRUE)

        if ((timeout != -1) && (gTxLOUnlocked == 0)) {
            return SMAC_SUCCESS;
        } // timeout
        else {
            return SMAC_FAILURE;
        } // timedout

    }
    else {
        SMAC_LOG_DEBUG_STATE(gi8AppStatus, 0x77);
        return RX_ON;
    }
}

/*!
 * \fn void PDDataIndication()
 *
 * \brief Receive data packet indication.
 *
 */
void PDDataIndication()
{
    HPRF_PROFILE(RxStart);
    /* Read the Data only if it is a good packet. */
    if (psDrvRxPacket->u8Status == SMAC_SUCCESS)
    {
        (void)RAMDrvReadRx(psDrvRxPacket);        /* 
                                             * Read data from MC13192, 
                                             * check status 
                                             */
    }
    EnableInterrupts;       /* Allow system interrupts within the IRQ handler */
    
    #if (SMAC_FEATURE_SECURITY == TRUE)
      SECSecure(psDrvRxPacket->pu8Data, psDrvRxPacket->u8DataLength);
    #endif /* SMAC_FEATURE_SECURITY */
    
    #if (SMAC_FEATURE_OTAP == TRUE)
      if (gu8OTAPStatusAPI == OTAP_ACTIVE){
        OTAPDataIndication(psDrvRxPacket);
      } else {
        MCPSDataIndication(psDrvRxPacket);
      }
    #else
      MCPSDataIndication(psDrvRxPacket);
    #endif /* SMAC_FEATURE_OTAP */

}

/*!
 * \fn UINT8 PLMEHibernateRequest(void) 
 *
 * \brief Hibernate the MC13192 (very low current, no CLKO).
 *
 * \return Status
 */
 UINT8 PLMEHibernateRequest(void)
{
    UINT16 u16CurrentValue;
    
    gu8RTxMode = HIBERNATE_MODE;
    u16CurrentValue = SPIDrvRead(MODE2_ADDR);       /* 
                                                     * Read MC13192 Hiberate 
                                                     * register. 
                                                     */
    u16CurrentValue &= 0xFFFC;
    u16CurrentValue |= 0x0002;                      /* Hiberate enable */
    SPIDrvWrite(MODE2_ADDR, u16CurrentValue);       /* 
                                                     * Write back to MC13192 to 
                                                     * enable hibernate mode. 
                                                     */
    return SMAC_SUCCESS;         /* Call back to MAC layer indicating success. */
}

/*!
 * \fn UINT8 PLMEDozeRequest(UINT32 u32Timeout) 
 *
 * \brief Doze the MC13192 (Low current, CLKO <= 1MHz).
 *
 * \param u32Timeout - timeout
 *
 * \return Status
 */

UINT8 PLMEDozeRequest(UINT32 u32Timeout)
{
    UINT16 u16CurrentValue;
    UINT32 u32CurrentTime;
    
    gu8RTxMode = DOZE_MODE;
    u16CurrentValue = SPIDrvRead(MODE2_ADDR);       /* 
                                                     * Read MC13192 Doze 
                                                     * register. 
                                                     */
    if (u32Timeout == 0)        /* ACOMA mode, with clkout */
    {
        u16CurrentValue = SPIDrvRead(IRQ_MASK);
        u16CurrentValue &= 0xFE0D;   /* Change the IRQ_Mask to set Acoma en,*/
        u16CurrentValue |= 0x0100;   /* doze irq disabled and tmr2 disabled*/
        SPIDrvWrite(IRQ_MASK, u16CurrentValue);      
        
        u16CurrentValue = SPIDrvRead(MODE2_ADDR);       /* 
                                                         * Read MC13192 Doze 
                                                         * register. 
                                                         */
        u16CurrentValue &= 0xFDFC;
        u16CurrentValue |= 0x0201;                      /* Doze (acoma) & CLKOUT enable */
        SPIDrvWrite(MODE2_ADDR, u16CurrentValue);       /* 
                                                         * Write back to MC13192 to 
                                                         * enable hibernate mode. 
                                                         */
    } else						 /* DOZE mode with timeout, no clkout */
    {
        u16CurrentValue = SPIDrvRead(IRQ_MASK);
        u16CurrentValue &= 0xFE0D;   /* Change the IRQ_Mask to set Acoma dis,*/
        u16CurrentValue |= 0x0012;   /* doze enabled and tmr2 enabled*/
        SPIDrvWrite(IRQ_MASK, u16CurrentValue);      
        
        u32CurrentTime = PLMEGetTimeRequest();
        u32Timeout += u32CurrentTime;
        SPIDrvWrite( T2_HI_ADDR, ((UINT16)(u32Timeout>>16)&0x00FF) ); 
        SPIDrvWrite( T2_LO_ADDR, ((UINT16)u32Timeout)&0xFFFF );  /* Enable the Timer 2 
                                                         * and save the timeout value
                                                         */
        u16CurrentValue = SPIDrvRead(MODE2_ADDR);
        u16CurrentValue &= 0xFDFC;						/* Disable CLKOinDozeand */
        u16CurrentValue |= 0x0001;                      /* enter in Doze mode */
        SPIDrvWrite(MODE2_ADDR, u16CurrentValue);      
    }
    return SMAC_SUCCESS;         /* Call back to MAC layer indicating success. */
}

 /*!
 * \fn UINT8 PLMEWakeRequest(void) 
 *
 * \brief Wake the MC13192 from Hibernate or Doze. 
 *
 * \return Status
 */
UINT8 PLMEWakeRequest(void)
{
    UINT16 u16CurrentValue;
    
    MC13192Wake();                              /* Wake up the device */
    while (gu8RTxMode != IDLE_MODE_ATTN)
    {
        MCU_LOW_POWER_WHILE();                    /* Wait until ATTN */
    }
    u16CurrentValue = SPIDrvRead(MODE2_ADDR);       /* 
                                                     * Read MC13192 
                                                     * Hibernate/Doze register. 
                                                     */
    u16CurrentValue &= 0xFFFC;                  /* Hiberate and Doze disable */
    SPIDrvWrite(MODE2_ADDR, u16CurrentValue);       /* 
                                                     * Write back to MC13192 to 
                                                     * disable hibernate and doze 
                                                     * mode. 
                                                     */
    SPIDrvWrite(T2_HI_ADDR, 0x0000);                /* Disable Timer2
                                                     * To avoid a T2 int because
                                                     * of doze w/timeout
                                                     */
    gu8RTxMode = IDLE_MODE;
    return SMAC_SUCCESS;
}

 /*!
 * \fn UINT8 PLMESetChannelRequest(UINT8 u8Channel)
 *
 * \brief Set the MC13192 operating channel.
 *
 * \param u8Channel Channel number
 *
 * \return Status
 */
UINT8 PLMESetChannelRequest(UINT8 u8Channel)
{
    switch (u8Channel)
    {
    case 0x00:
        SPIDrvWrite(LO1_IDIV_ADDR,0x0F95);
        SPIDrvWrite(LO1_NUM_ADDR,0x5000);
    break;
    case 0x01:
        SPIDrvWrite(LO1_IDIV_ADDR,0x0F95);
        SPIDrvWrite(LO1_NUM_ADDR,0xA000); 
    break;
    case 0x02:
        SPIDrvWrite(LO1_IDIV_ADDR,0x0F95);
        SPIDrvWrite(LO1_NUM_ADDR,0xF000); 
    break;
    case 0x03:
        SPIDrvWrite(LO1_IDIV_ADDR,0x0F96);
        SPIDrvWrite(LO1_NUM_ADDR,0x4000); 
    break;
    case 0x04:
        SPIDrvWrite(LO1_IDIV_ADDR,0x0F96);
        SPIDrvWrite(LO1_NUM_ADDR,0x9000); 
    break;
    case 0x05:
        SPIDrvWrite(LO1_IDIV_ADDR,0x0F96);
        SPIDrvWrite(LO1_NUM_ADDR,0xE000); 
    break;
    case 0x06:
        SPIDrvWrite(LO1_IDIV_ADDR,0x0F97);
        SPIDrvWrite(LO1_NUM_ADDR,0x3000); 
    break;
    case 0x07:
        SPIDrvWrite(LO1_IDIV_ADDR,0x0F97);
        SPIDrvWrite(LO1_NUM_ADDR,0x8000); 
    break;
    case 0x08:
        SPIDrvWrite(LO1_IDIV_ADDR,0x0F97);
        SPIDrvWrite(LO1_NUM_ADDR,0xD000); 
    break;
    case 0x09:
        SPIDrvWrite(LO1_IDIV_ADDR,0x0F98);
        SPIDrvWrite(LO1_NUM_ADDR,0x2000); 
    break;
    case 0x0A:
        SPIDrvWrite(LO1_IDIV_ADDR,0x0F98);
        SPIDrvWrite(LO1_NUM_ADDR,0x7000); 
    break;
    case 0x0B:
        SPIDrvWrite(LO1_IDIV_ADDR,0x0F98);
        SPIDrvWrite(LO1_NUM_ADDR,0xC000); 
    break;
    case 0x0C:
        SPIDrvWrite(LO1_IDIV_ADDR,0x0F99);
        SPIDrvWrite(LO1_NUM_ADDR,0x1000); 
    break;
    case 0x0D:
        SPIDrvWrite(LO1_IDIV_ADDR,0x0F99);
        SPIDrvWrite(LO1_NUM_ADDR,0x6000); 
    break;
    case 0x0E:
        SPIDrvWrite(LO1_IDIV_ADDR,0x0F99);
        SPIDrvWrite(LO1_NUM_ADDR,0xB000); 
    break;
    case 0x0F:
        SPIDrvWrite(LO1_IDIV_ADDR,0x0F9A);
        SPIDrvWrite(LO1_NUM_ADDR,0x0000); 
    break;
    default:
        SPIDrvWrite(LO1_IDIV_ADDR,0x0F97);
        SPIDrvWrite(LO1_NUM_ADDR,0xD000); 
        return ERROR;
    }
    return SMAC_SUCCESS;
}

/*!
 * \fn UINT8 PLMESetTrxStateRequest(UINT8 u8ReqMode) 
 *
 * \brief Set the MC13192 transceive operation.
 *
 * \param u8ReqMode Operational mode
 *
 * \return Status  
 */
UINT8 PLMESetTrxStateRequest(UINT8 u8ReqMode)
{
  UINT16 u16Reg;

  RTXENDeAssert(); 
  u16Reg = SPIDrvRead(MODE_ADDR);
  u16Reg &= 0xFFF8;                           /* Clear mode. */
  switch (u8ReqMode)
  {
    //u16Reg |= IDLE_MODE;
  case IDLE_MODE:                         /* Write Idle */
#if defined (MC13192_LNA_PRESENT)
    MC13192_LNA_CTRL = LNA_OFF;
#endif
#if defined (MC13192_PA_PRESENT)
    MC1319xDrv_TxHighPowerDisable();
#endif

#ifdef RF_COMBINE_MODE	
    //	MC13202- configure rf switch mode 
    {
        volatile UINT16 u16Temp;
        u16Temp= SPIDrvRead(MODE2_ADDR);
        u16Temp |= (RF_SWITCH_MODE|CT_BIAS_EN);				
        SPIDrvWrite(MODE2_ADDR, u16Temp);      
    }
#endif

    gu8RTxMode = IDLE_MODE;
    SPIDrvWrite(MODE_ADDR, u16Reg);
    SMAC_LOG_DEBUG_STATE (gi8AppStatus, SMAC_SM_DEBUG_EVENT_IDLE_PLMESETSTATE);
    break;



  case RX_MODE:                           /* Write RX */
#if defined (MC13192_ANTENNA_SWITCH_PRESENT)
    MC1319xDrv_RxAntennaSwitchEnable();
#endif
#if defined (MC13192_LNA_PRESENT)
    MC13192_LNA_CTRL = LNA_ON;
#endif
#if defined (MC13192_PA_PRESENT)
    MC1319xDrv_TxHighPowerDisable();
#endif
#ifdef RF_COMBINE_MODE		
	
    // MC13202- configure rf switch mode 
    {
        volatile UINT16 u16Temp;
        u16Temp= SPIDrvRead(MODE2_ADDR);
        u16Temp |= (RF_SWITCH_MODE|CT_BIAS_EN);				
        SPIDrvWrite(MODE2_ADDR, u16Temp);      
    }
#endif
    gu8RTxMode = RX_MODE;
    u16Reg |= RX_MODE;
    SPIDrvWrite(MODE_ADDR, u16Reg);
    RTXENAssert(); 
    SMAC_LOG_DEBUG_STATE (gi8AppStatus, SMAC_SM_DEBUG_EVENT_RX_PLMESETSTATE);
    break;




  case RX_MODE_WTO:           /* Write RX, but set gu8RTxMode to timeout */
#if defined (MC13192_ANTENNA_SWITCH_PRESENT)
    MC1319xDrv_RxAntennaSwitchEnable();
#endif
#if defined (MC13192_LNA_PRESENT)
    MC13192_LNA_CTRL = LNA_ON;
#endif
#if defined (MC13192_PA_PRESENT)
    MC1319xDrv_TxHighPowerDisable();
#endif
#ifdef RF_COMBINE_MODE	
    //	MC13202- configure rf switch mode 
    {
        volatile UINT16 u16Temp;
        u16Temp= SPIDrvRead(MODE2_ADDR);
        u16Temp |= (RF_SWITCH_MODE|CT_BIAS_EN);				
        SPIDrvWrite(MODE2_ADDR, u16Temp);      
    }
#endif
    gu8RTxMode = RX_MODE_WTO;
    u16Reg |= RX_MODE;
    SPIDrvWrite(MODE_ADDR, u16Reg);
    RTXENAssert();
    SMAC_LOG_DEBUG_STATE (gi8AppStatus, SMAC_SM_DEBUG_EVENT_RXWTO_PLMESETSTATE);
    break;




  case TX_MODE:               /* Write Tx. Note: force LO lock not used */
#if defined (MC13192_PA_PRESENT)
    MC1319xDrv_TxHighPowerEnable();
#endif
#if defined (MC13192_LNA_PRESENT)
    MC13192_LNA_CTRL = LNA_OFF;
#endif
#if defined (MC13192_ANTENNA_SWITCH_PRESENT)
    MC1319xDrv_TxAntennaSwitchEnable();
#endif
#ifdef RF_COMBINE_MODE	
    //	MC13202- configure rf switch mode 
    {
        volatile UINT16 u16Temp;
        u16Temp= SPIDrvRead(MODE2_ADDR);	
        //if (Rf_Smac_802_PHYParams.outputPower < 19)	{			
        if (1)	{   // leave in combine mode for all power levels (TEST)
            // for low power level use combine transmit/receive pins
            u16Temp |= (RF_SWITCH_MODE|CT_BIAS_EN);				
            SPIDrvWrite(MODE2_ADDR, u16Temp);
        } else {
            // for higher power level use separate transmit pins
            u16Temp &= ~(RF_SWITCH_MODE);				
            SPIDrvWrite(MODE2_ADDR, u16Temp);
        }
    }
#endif

    u16Reg |= TX_MODE;
    gu8RTxMode = TX_MODE;
    SPIDrvWrite(MODE_ADDR, u16Reg);
    RTXENAssert();
    SMAC_LOG_DEBUG_STATE (gi8AppStatus, SMAC_SM_DEBUG_EVENT_TX_PLMESETSTATE);
    break;




  case PULSE_TX_MODE: /* Write Tx. Note: force LO lock not used */

#if defined (MC13192_ANTENNA_SWITCH_PRESENT)
    MC1319xDrv_TxAntennaSwitchEnable();
#endif
#if defined (MC13192_PA_PRESENT)
    MC1319xDrv_TxHighPowerEnable();
#endif
#if defined (MC13192_LNA_PRESENT)
    MC13192_LNA_CTRL = LNA_OFF;
#endif

#ifdef RF_COMBINE_MODE	
    //	MC13202- configure rf switch mode 
    {
        volatile UINT16 u16Temp;
        u16Temp= SPIDrvRead(MODE2_ADDR);
        //if (Rf_Smac_802_PHYParams.outputPower < 19)	{	
        if (1) {    // leave in combine mode for all power levels (TEST)
            // for low power level use combine transmit/receive pins
            u16Temp |= (RF_SWITCH_MODE|CT_BIAS_EN);				
            SPIDrvWrite(MODE2_ADDR, u16Temp);
        } else {
            // for higher power level use separate transmit pins
            u16Temp &= ~(RF_SWITCH_MODE);				
            SPIDrvWrite(MODE2_ADDR, u16Temp);
        }
    }
#endif
    u16Reg |= TX_MODE;
    gu8RTxMode = PULSE_TX_MODE;
    SPIDrvWrite(MODE_ADDR, u16Reg);
    RTXENAssert();
    break;

  default:
    return ERROR;    
  }
  return SMAC_SUCCESS;
}

/*!
 * \fn UINT8 PLMEEnergyDetect (void) 
 *
 * \brief Measure channel energy.
 *
 * \return Energy
 * \note Actual power returned is: -(power/2)   \n
 * Global calibration required for accuracy. 
 */
UINT8 PLMEEnergyDetect (void)
{
    UINT16 u16Reg;
    UINT8 u8Power;
    
    gu8RTxMode = CCA_MODE;                /* Write energy detect mode */
    u16Reg = SPIDrvRead(MODE_ADDR);
    
    u16Reg &= 0xFFC8; /* Mask off the CCA_type mode 5:4, xcvr_seq 2:0 */    
    u16Reg |= (CCA_MODE | SCAN_MODE_ED); /* Set the CCA mode and turn on CCA */
    SPIDrvWrite(MODE_ADDR, u16Reg);
    RTXENAssert();
    while (gu8RTxMode != IDLE_MODE)       /* Wait for energy detect to complete */
        {
        MCU_LOW_POWER_WHILE();
        }
    u16Reg = (SPIDrvRead(CCA_RESULT_ADDR) & 0xFF00);
    u8Power = (UINT8)(u16Reg >> 8);
    return u8Power;
}

UINT8 PLMECCARequest (void)
/* Note: Actual power returned is: -(power/2) */
/* Global calibration required for accuracy. */
{
    UINT16 u16Reg;
    //UINT8 u8Power;
    // MNT - 7/2/2007 - Added a timeout so that we don't stay 
    // waiting for the state forever. Currently, we are getting watchdog
    // resets. 
    // MNT - 7/3/2007 - THis timeout is too long. A CCA operation will take approximately
    // 280 us. So, setting a timeout of 2560 us is excessive. Change this so that
    // the timeout is set to 350 us
    INT32 timeout = 35;		// PRASAD (working as of now, may need to change?)
    
#ifdef RF_COMBINE_MODE	
    // MC13202- enable combine receiver mode
    {
        volatile UINT16 u16Temp;
        u16Temp= SPIDrvRead(MODE2_ADDR);
        u16Temp |= (RF_SWITCH_MODE|CT_BIAS_EN);				
        SPIDrvWrite(MODE2_ADDR, u16Temp);      
    }
#endif
    gu8RTxMode = CCA_MODE;                /* Write energy detect mode */
    u16Reg = SPIDrvRead(MODE_ADDR);
    
    u16Reg &= 0xFFC8; /* Mask off the CCA_type mode 5:4, xcvr_seq 2:0 */    
    u16Reg |= (CCA_MODE | SCAN_MODE_CCA); /* Set the CCA mode and turn on CCA */
	SPIDrvWrite(MODE_ADDR, u16Reg);
    RTXENAssert();
//    HwDelayUsec(350); //TODO: Revisit
    while ((gu8RTxMode != IDLE_MODE) && (timeout--))      /* Wait for energy detect to complete */
    {
		MCU_LOW_POWER_WHILE();
    }

    // MNT - 7/2/2007 - If we timed out, just return
    if (timeout <= -1)
    {
      // MNT - 7/3/2007 - Force Idle mode - this will allow us to do another CCA if needed
	  Rfdiags_NumCCATimeoutsFailures++;
      RTXENDeAssert();
      // MNT - 10/3/2007 - Count the number of CCA failures that occured here
      Rfdiags_NumCCAFailures++;
      return 1; // return busy so we can try again
    } //if (timeout <= -1)
    
    if ((g_u16StatusContent & CCA_BIT_MASK) == 0)
	{
      return 0; //Clear  
    }
	else
	{
      // MNT - 10/3/2007 - Count the number of CCA failures that occured here
      Rfdiags_NumCCAFailures++;
      return 1; //BUSY
    }
}

/*!
 * \fn UINT8 PLMELinkQuality (void)
 *
 * \brief Report energy from last successful RX packet.
 *
 * \return Energy
 *
 * \note Actual power returned is: -(power/2) \n
 *     Global calibration required for accuracy. 
 */
UINT8 PLMELinkQuality (void)
{
    UINT16 u16Reg;
    UINT8 u8Power;
    
    u16Reg = SPIDrvRead(CCA_RESULT_ADDR);
    u8Power = (UINT8)((u16Reg & 0xFF00) >> 8);
    return u8Power;
}

/*!
 * \fn UINT32 PLMEGetTimeRequest(void) 
 *
 * \brief Get MC13192 timer value. 
 *
 * \return Timer value
 */

UINT32 PLMEGetTimeRequest(void)
{
    UINT32 u32UpperWord, u32LowerWord;
    UINT32 u32CurrentTime;
    
    DisableInterrupts;
    u32UpperWord = SPIDrvRead(TIMESTAMP_HI_ADDR);
    u32LowerWord = SPIDrvRead(TIMESTAMP_LO_ADDR);
    u32UpperWord &= TIMESTAMP_HI_MASK;              /* Clears TS_HELD bit. */
    u32CurrentTime = (UINT32) (u32UpperWord << 16) | u32LowerWord;
    EnableInterrupts;
    return u32CurrentTime;
}

/*!
 * \fn UINT8 PLMESetMC13192ClockRate(UINT8 u8Freq)
 *
 * \brief Set MC13192 CLKo frequency.
 *
 * \param u8Freq Frequency value
 *
 * \return Status
 */

UINT8 PLMESetMC13192ClockRate(UINT8 u8Freq)
{
    volatile UINT16 u16CurrentValue;
    
    u16CurrentValue = SPIDrvRead(CLKS_ADDR); /* Read register and re-write */
    u16CurrentValue &= 0xFFF8;
    u16CurrentValue |= u8Freq;
    SPIDrvWrite(CLKS_ADDR, u16CurrentValue);
    return SMAC_SUCCESS;
}

/*!
 * \fn UINT8 PLMESetMC13192TmrPrescale (UINT8 freq) 
 *
 * \brief Set MC13192 timer frequency.
 *
 * \param freq Frequency value
 *
 * \return Status
 */
UINT8 PLMESetMC13192TmrPrescale (UINT8 freq) 
{
    volatile UINT16 current_value;
    
    current_value = SPIDrvRead(PRESCALE_ADDR);
    current_value &= 0xFFF8;
    current_value |= freq;
    SPIDrvWrite(PRESCALE_ADDR, current_value);
    return SMAC_SUCCESS;
}

/*!
 * \fn void PLMESetTimeRequest(UINT32 u32RequestedTime)
 *
 * \brief Set MC13192 timer value (i.e. initialize)
 *
 * \param u32RequestedTime Timer Value
 *
 */
void PLMESetTimeRequest(UINT32 u32RequestedTime)
{
    UINT16 u16UpperWord, u16LowerWord;
    UINT16 u16Mode2RegVal;
    
    /* Split 32 bit input into 2 16 bit values */
    u16UpperWord = (UINT16) (u32RequestedTime >> 16) & 0x000000FF;
    u16LowerWord = (UINT16) u32RequestedTime & 0x0000FFFF;
    /* Program Time1 comparator with the desired value */    
    SPIDrvWrite(T1_HI_ADDR, u16UpperWord);
    SPIDrvWrite(T1_LO_ADDR, u16LowerWord);
    /* Get current state of the MODE2 MC13192 register */
    u16Mode2RegVal = SPIDrvRead(MODE2_ADDR);
    /* Set the Tmr_load bit */
    u16Mode2RegVal |= 0x8000;
    /* Now write the value back to MC13192 register MODE2 */
    SPIDrvWrite(MODE2_ADDR, u16Mode2RegVal);
    /* Clear the tmr_load bit */
    u16Mode2RegVal &= 0x7FFF;
    /* Clr the tmr_load bit to prepare for next set_time_request. */
    SPIDrvWrite(MODE2_ADDR, u16Mode2RegVal);    
    return;
}

/*!
 * \fn UINT8 PLMEEnableMC13192Timer1(UINT32 u32CounterValue)
 *
 * \brief Set MC13192 timer compare value.
 *
 * \param u32CounterValue 
 *
 * \return Status
 */
UINT8 PLMEEnableMC13192Timer1(UINT32 u32CounterValue)
{
    /* Load the timeout value into T1 with Timer disabled. */
    SPIDrvWrite(T1_HI_ADDR, (UINT16) ((u32CounterValue >> 16) & 0x000000FF)|\
                                           0x000080FF);
    SPIDrvWrite(T1_LO_ADDR, (UINT16) (u32CounterValue & 0x0000FFFF));
    /* Turn Timer1 mask on. */
    SPIDrvWrite(T1_HI_ADDR, (UINT16) ((u32CounterValue >> 16) & 0x000000FF));
    SPIDrvWrite(T1_LO_ADDR, (UINT16) (u32CounterValue & 0x0000FFFF));
    return SMAC_SUCCESS;
}

 /*!
 * \fn UINT8 PLMEDisableMC13192Timer1(void) 
 *
 * \brief Disable MC13192 timer comparator TC1.
 *
 * \return Status
 */
UINT8 PLMEDisableMC13192Timer1(void)
{
    /* Load the timeout value into T1 with Timer disabled. */
    /* Clear Timer1 if in RX_MODE_WTO */
    SPIDrvWrite(T1_HI_ADDR, 0x8000);
    SPIDrvWrite(T1_LO_ADDR, 0x0000);
    //irq_mask_reg = SPIDrvRead(IRQ_MASK);      /* NOT USED */
    //irq_mask_reg &= ~TIMER1_IRQMASK_BIT;
    //SPIDrvWrite(IRQ_MASK, irq_mask_reg);
    return SMAC_SUCCESS;
}

/*!
 * \fn void PLMEMC13192ResetIndication (void) 
 *
 * \brief Indicate a MC13192 reset condition.
 *
 */
void PLMEMC13192ResetIndication (void)
{
    MLMEMC13192ResetIndication();
}

/*!
 * \brief Force the MC13192 into a soft reset condition.
 *
 * \return Status 
 */
UINT8 PLMEMC13192SoftReset(void)
{
    SPIDrvWrite(RESET_REG, 0x00);
    return SMAC_SUCCESS;
}

/*!
 * \fn UINT8 PLMEMC13192XtalAdjust(UINT8 u8TrimValue) 
 *
 * \brief Adjust the MC13192s crystal trim value.
 *
 * \param u8TrimValue Trim value
 *
 * \return Status
 */
UINT8 PLMEMC13192XtalAdjust(UINT8 u8TrimValue)
{
    UINT16 u16Reg;
    UINT16 u16RegValue;
    
    u16RegValue = (UINT16)(u8TrimValue << 8);/* 
                                              * Shift the req value into the 
                                              * higher half word 
                                              */
    u16Reg = SPIDrvRead(XTAL_ADJ_ADDR);      /* 
                                              * Read the current value of XTAL 
                                              * Reg 
                                              */
    u16Reg = ((u16Reg & 0x00FF) | u16RegValue);
    SPIDrvWrite(XTAL_ADJ_ADDR, u16Reg);
    return SMAC_SUCCESS;
}

/*!
 * \fn UINT8 PLMEMC13192FEGainAdjust(UINT8 i8GainValue)
 *
 * \brief Adjust the MC13192s gain compensator.
 *
 * \param i8GainValue Gain Compensation
 *
 * \return Status
 */
UINT8 PLMEMC13192FEGainAdjust(UINT8 i8GainValue)
{
    UINT16 u16Reg;
    
    u16Reg = SPIDrvRead(FEGAIN_ADDR);       /* 
                                             * Read the current value of GAIN 
                                             * Reg 
                                             */
    u16Reg = ((u16Reg & 0xFF00) | i8GainValue);
    SPIDrvWrite(FEGAIN_ADDR, u16Reg);
    return SMAC_SUCCESS;
}

 /*!
 * \fn UINT8 PLMEMC13192PAOutputAdjust(UINT8 u8RequestedPAValue)
 *
 * \brief Adjust the MC13192s Output power.
 *
 * \param u8RequestedPAValue PA output adjust
 *
 * \return Status
 */
UINT8 PLMEMC13192PAOutputAdjust(UINT8 u8RequestedPAValue)
{
    UINT16 u16Reg;
    UINT8 u8PAValue;
    
    switch (u8RequestedPAValue)
    {
        case MAX_POWER:         /* Sets the PA drive level and PA gain to MAX */
            u8PAValue = 0xFF;
        break;

        case MIN_POWER:
            u8PAValue = 0x00;   /* Sets the PA drive level and PA gain to MIN */
        break;

        default:
            // MNT - 7/13/07 - Increased the number of steps from 16 to 22
            // Freescale added 6 levels - 3 above and 3 below the normal 16 values.
            // This number along with the others in this function are hardcoded with
            // comments to make it a little more readable.
            if (u8RequestedPAValue > (21)) {
                return SMAC_OVERFLOW;
            }
            else {                
                u8PAValue = u8RequestedPAValue;
            }
        break;
    }
    
    u16Reg = SPIDrvRead(PA_ADJUST_ADDR);    /* 
                                             * Read the current value of GAIN 
                                             * Register 
                                             */
    u16Reg &= 0xFF00;
    
    if ((u8RequestedPAValue == MAX_POWER) || (u8RequestedPAValue == MIN_POWER)){
        u16Reg |= u8PAValue;
    }
    else {

      
        // MNT - 7/13/07 - Increased the number of steps from 16 to 22
        // Here's the table from the MC13192 manual - The values above
        // 18 are not allowed to be set due to FCC requirements.
      /*
		              PA Power Adjust	  Typical Differential Power at
        Parameter	Reg 12[7:0]		    Output Contact (dBm)
	                (Hex)	
        0	        00	              -27.6
        1	        04	              -20.6
        2	        08	              -17.7
        3	        0C	              -16.3
        4	        1C	              -15.7
        5	        2C	              -15.2
        6	        3C	              -14.6
        7	        4C	              -8.9
        8	        5C	              -8.2
        9	        6C	              -7.5
        10	      7C	              -7.1
        11	      8C	              -1.6
        12	      9C	              -1.1
        13	      AC	              -0.7
        14	      BC                (default)	-0.3
        15	      CC	              1.3
        16	      DC	              1.9
        17	      EC	              2.5
        18	      FC	              2.6
        19	      FD	              3.2
        20	      FE	              3.7
        21	      FF	              4.1
      */


      // u8RequestedPAValue can only be between 0 and 21 based on the check above

      if (u8RequestedPAValue < 4)
      {
        // upper 2 nibbles are 0, lower nibbles multiple of 4
        u16Reg = u8RequestedPAValue*4;
      } // if (u8RequestedPAValue < 4)
      else if (u8RequestedPAValue > 17)
      {
        // Lower nibble goes from 12 to 15, upper nibble is set to 15
        u16Reg =  ((0xF << 4) | (u8RequestedPAValue-6));
      } //else if (u8RequestedPAValue > 17)
      else
      {
        u16Reg |= (((u8PAValue-3) << 4) | 0x000C);
      }//else if (u8RequestedPAValue > 17)


    }
    SPIDrvWrite(PA_ADJUST_ADDR, u16Reg);
    return SMAC_SUCCESS;
}

/*!
 * \fn UINT8 PLMEGetRficVersion(void)
 *
 * \brief Returns the RFIC version number.
 *
 * \return Version number 
 */
UINT16 PLMEGetRficVersion(void)
{
    UINT16 u16Reg;
    
    u16Reg = SPIDrvRead(VERSION_REG);   /* 
                                         * Read the version register 
                                         * version[12:10] 
                                         */
    u16Reg &= VERSION_MASK;             /* Shift to generate accurate number */
    
    return (u16Reg);
}
/*!
 * \fn void PLMELoadPRBS9 (tTxPacket *psPacket)
 *
 * \brief Loads the transmit RAM with a PRBS9 data pattern.
 *
 * \param  *psPacket Transmit buffer pointer
 *
 */
void PLMELoadPRBS9 (tTxPacket *psPacket)
{
  UINT8 u8c1;                           /* Byte counter */
  UINT8 u8c2;                           /* Bit counter */
  UINT16 u16t1;                         /* LFSR */
  UINT16 u16t2;                         /* LFSR output */
  UINT16 u16t3;                         /* LFSR feedback tap */
  UINT8 u8t4;                           /* Assembled transmit byte */
  
    u16t1 = 0x01FF;                     /* Initialize the LFSR */
    for (u8c1=0; u8c1<64; u8c1++)       /* Byte counter */
    {
        u8t4 = 0x00;                    /* Initialize the byte */
    for (u8c2=0; u8c2<8; u8c2++)        /* Bit counter */
    {
        u16t2 = (u16t1 & 0x0001);       /* LFSR output */
        if (u16t2 == 0x0001)
        {
            u8t4 = (UINT8)(u8t4 | 0x80);         /* Set/Clear byte based on LFSR output */
        }
        if (u8c2 != 7)
        {
            u8t4 = (UINT8)(u8t4 >> 1);           /* LSBit will be first bit out of LFSR */
        }
        u16t3 = ((u16t1 & 0x0010) >> 4);/* LFSR tap */
        u16t1 = u16t1 >> 1;             /* Now shift the LFSR */
        if (u16t2 == u16t3)             /* Set/Clr the LFSR MSBit */
        {
            u16t1 = u16t1 & 0xFEFF;
        }
        else
        {
            u16t1 = u16t1 | 0x0100;
        }
    }
    psPacket->pu8Data[u8c1] = u8t4;
    }
    psPacket->u8DataLength = 64;
    RAMDrvWriteTx(psPacket);            /* Load the data into packet RAM */
}

/*!
 * \fn void PLMELoadPRBS11 (tTxPacket *psPacket)
 *
 * \brief Loads the transmit RAM with a PRBS11 data pattern.
 * Polynomial: x^11 + x^9 + 1
 * http://en.wikipedia.org/wiki/Linear_feedback_shift_register
 *
 * \param  *psPacket Transmit buffer pointer
 *
 */
void PLMELoadPRBS11 (tTxPacket *psPacket)
{
    UINT16 u8c1;            /* Byte counter */
    UINT8 u8c2;             /* Bit counter */
    UINT16 u16t1;           /* LFSR */
    UINT16 u16t2;           /* LFSR output */
    UINT16 u16t3;           /* LFSR feedback tap */
    UINT8 u8t4;             /* Assembled transmit byte */
    
    UINT8 buffer[260];      /* This is a 2048 bit sequence, or 256 bytes, we can't load the psPacket (max is 128) */

  
    u16t1 = 0x07FF;                             /* Initialize the LFSR with seed, all 1's */
    for (u8c1=0; u8c1<256; u8c1++)              /* Byte counter */
    {
        u8t4 = 0x00;                            /* Initialize the byte */
        for (u8c2=0; u8c2<8; u8c2++)            /* Bit counter */
        {
            u16t2 = (u16t1 & 0x0001);           /* LFSR output */
            if (u16t2 == 0x0001)
            {
                u8t4 = (UINT8)(u8t4 | 0x80);    /* Set/Clear byte based on LFSR output */
            }
            if (u8c2 != 7)
            {
                u8t4 = (UINT8)(u8t4 >> 1);      /* LSBit will be first bit out of LFSR */
            }
            u16t3 = ((u16t1 & 0x0004) >> 2);    /* LFSR tap */
            u16t1 = u16t1 >> 1;                 /* Now shift the LFSR */
            if (u16t2 == u16t3)                 /* Set/Clr the LFSR MSBit */
            {
                u16t1 = u16t1 & 0xFBFF;
            }
            else
            {
                u16t1 = u16t1 | 0x0400;
            }
        }
        //psPacket->pu8Data[u8c1] = u8t4;
        buffer[u8c1] = u8t4;
    }

    psPacket->u8DataLength = 124;       /* just take first 124 bytes in the sequence */
    for (u8c1=0; u8c1<124; u8c1++)
        psPacket->pu8Data[u8c1] =  buffer[u8c1];

    RAMDrvWriteTx(psPacket);            /* Load the data into packet RAM */
}

///////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////
#if 0   /* Following PRBS we are not using */
/*!
 * \fn void PLMELoadPRBS8 (tTxPacket *psPacket)
 *
 * \brief Loads the transmit RAM with a PRBS8 data pattern.
 * Polynomial: x^8 + x^6 + x^5 + x^4 + 1
 * http://en.wikipedia.org/wiki/Linear_feedback_shift_register
 *
 * \param  *psPacket Transmit buffer pointer
 *
 */
void PLMELoadPRBS8 (tTxPacket *psPacket)
{
    UINT8 ByteCounter;              /* Byte counter */
    UINT8 BitCounter;               /* Bit counter */
    UINT8 LFSR;                     /* LFSR */
    UINT8 LFSROut;                  /* LFSR output */
    UINT8 TapFeedback;              /* LFSR feedback tap */
    UINT8 TxByte;                   /* Assembled transmit byte */

    /*int i = 0, j = 0;
    UINT8 u8Data;*/
  
    LFSR = 0xFF;                    /* Initialize the LFSR */
    for (ByteCounter=0; ByteCounter<32; ByteCounter++)          /* Byte counter */
    {
        TxByte = 0x00;                                          /* Initialize the byte */
        for (BitCounter=0; BitCounter<8; BitCounter++)          /* Bit counter */
        {
            LFSROut = (LFSR & 0x01);                            /* LFSR output */
            if (LFSROut == 0x01)
            {
                TxByte = (UINT8)(TxByte | 0x80);                /* Set/Clear byte based on LFSR output */
            }
            if (BitCounter != 7)
            {
                TxByte = (UINT8)(TxByte >> 1);                  /* LSBit will be first bit out of LFSR */
            }
            TapFeedback = (((LFSR & 0x01) ^ ((LFSR >> 2) & 0x01)) ^ ((LFSR >> 3) & 0x01)) ^ ((LFSR >> 4) & 0x01);   /* LFSR tap */
            LFSR = LFSR >> 1;                                   /* Now shift the LFSR */
            if (TapFeedback == 0x01)                            /* Set/Clr the LFSR MSBit */
            {
                LFSR = LFSR | 0x80;
            }
            else
            {
                LFSR = LFSR & 0x7F;
            }
        }
        psPacket->pu8Data[ByteCounter] = TxByte;
    }

    psPacket->u8DataLength = 32;

    /*ConsolePrintf("PRBS8 Sequence: \r\n");
    for (i=0; i<32; i++)
    {
        u8Data = psPacket->pu8Data[i];
        for(j = 0; j < 8; j++)
        {
            ConsolePrintf("%d", u8Data & 0x01);
            u8Data = u8Data >> 1;
        }
    }
    ConsolePrintf("\r\n\r\n");*/

    RAMDrvWriteTx(psPacket);            /* Load the data into packet RAM */
}

/*!
 * \fn void PLMELoadPRBS10 (tTxPacket *psPacket)
 *
 * \brief Loads the transmit RAM with a PRBS10 data pattern.
 * Polynomial: x^10 + x^7 + 1
 * http://en.wikipedia.org/wiki/Linear_feedback_shift_register
 *
 * \param  *psPacket Transmit buffer pointer
 *
 */
void PLMELoadPRBS10 (tTxPacket *psPacket)
{
    UINT8 u8c1;                           /* Byte counter */
    UINT8 u8c2;                           /* Bit counter */
    UINT16 u16t1;                         /* LFSR */
    UINT16 u16t2;                         /* LFSR output */
    UINT16 u16t3;                         /* LFSR feedback tap */
    UINT8 u8t4;                           /* Assembled transmit byte */

    /*int i = 0, j = 0;
    UINT8 u8Data;*/
  
    u16t1 = 0x03FF;                                 /* Initialize the LFSR */
    for (u8c1=0; u8c1<128; u8c1++)                  /* Byte counter */
    {
        u8t4 = 0x00;                                /* Initialize the byte */
        for (u8c2=0; u8c2<8; u8c2++)                /* Bit counter */
        {
            u16t2 = (u16t1 & 0x0001);               /* LFSR output */
            if (u16t2 == 0x0001)
            {
                u8t4 = (UINT8)(u8t4 | 0x80);        /* Set/Clear byte based on LFSR output */
            }
            if (u8c2 != 7)
            {
                u8t4 = (UINT8)(u8t4 >> 1);          /* LSBit will be first bit out of LFSR */
            }
            u16t3 = ((u16t1 & 0x0008) >> 3);        /* LFSR tap */
            u16t1 = u16t1 >> 1;                     /* Now shift the LFSR */
            if (u16t2 == u16t3)                     /* Set/Clr the LFSR MSBit */
            {
                u16t1 = u16t1 & 0xFDFF;
            }
            else
            {
                u16t1 = u16t1 | 0x0200;
            }
        }
        psPacket->pu8Data[u8c1] = u8t4;
    }

    psPacket->u8DataLength = 128;

    /*ConsolePrintf("PRBS10 Sequence: \r\n");
    for (i=0; i<128; i++)
    {
        u8Data = psPacket->pu8Data[i];
        for(j = 0; j < 8; j++)
        {
            ConsolePrintf("%d\r\n", u8Data & 0x01);
            u8Data = u8Data >> 1;
        }
    }
    ConsolePrintf("\r\n\r\n");*/

    RAMDrvWriteTx(psPacket);            /* Load the data into packet RAM */
}

/*!
 * \fn void PLMELoadPRBS7 (tTxPacket *psPacket)
 *
 * \brief Loads the transmit RAM with a PRBS7 data pattern.
 * Polynomial: x^7 + x^7 + 1
 * http://en.wikipedia.org/wiki/Linear_feedback_shift_register
 *
 * \param  *psPacket Transmit buffer pointer
 *
 */
void PLMELoadPRBS7 (tTxPacket *psPacket)
{
    UINT8 ByteCounter;            /* Byte counter */
    UINT8 BitCounter;             /* Bit counter */
    UINT8 LFSR;                   /* LFSR */
    UINT8 LFSROut;                /* LFSR output */
    UINT8 TapFeedback;            /* LFSR feedback tap */
    UINT8 TxByte;                 /* Assembled transmit byte */

    /*int i = 0, j = 0;
    UINT8 u8Data;*/
  
    LFSR = 0x7F;                  /* Initialize the LFSR - seed with all 1's */
    for (ByteCounter=0; ByteCounter<16; ByteCounter++)       /* Byte counter */
    {
        TxByte = 0x00;                                      /* Initialize the byte */
        for (BitCounter=0; BitCounter<8; BitCounter++)      /* Bit counter */
        {
            LFSROut = (LFSR & 0x01);                        /* LFSR output */
            if (LFSROut == 0x01)
            {
                TxByte = (UINT8)(TxByte | 0x80);            /* Set/Clear byte based on LFSR output */
            }
            if (BitCounter != 7)
            {
                TxByte = (UINT8)(TxByte >> 1);              /* LSBit will be first bit out of LFSR */
            }
            TapFeedback = (LFSR & 0x01) ^ ((LFSR >> 1) & 0x01);     /* LFSR tap */
            LFSR = LFSR >> 1;                                       /* Now shift the LFSR */
            if (TapFeedback == 0x01)                                /* Set/Clr the LFSR MSBit */
            {
                LFSR = LFSR | 0x40;
            }
            else
            {
                LFSR = LFSR & 0xBF;
            }
        }
        psPacket->pu8Data[ByteCounter] = TxByte;
    }
    psPacket->u8DataLength = 16;

    /*ConsolePrintf("PRBS7 Sequence: \r\n");
    for (i=0; i<16; i++)
    {
        u8Data = psPacket->pu8Data[i];
        for(j = 0; j < 8; j++)
        {
            ConsolePrintf("%d", u8Data & 0x01);
            u8Data = u8Data >> 1;
        }
    }
    ConsolePrintf("\r\n\r\n");*/

    RAMDrvWriteTx(psPacket);            /* Load the data into packet RAM */
}
#endif  /* #if 0 */
////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////

UINT8  PLMEEnablePromiscuousMode(void)
{
//  gsPhyOptions.Bits.u1PromiscuousMode = TRUE;
  return SMAC_SUCCESS;
}
UINT8  PLMEDisablePromiscuousMode(void)
{
//  gsPhyOptions.Bits.u1PromiscuousMode = FALSE;
  return SMAC_SUCCESS;
}



UINT8   PLMESetCCAThreshPowerComp(UINT8 CCAThresh, UINT8 PowerComp)
{
  UINT16 u16Reg = ((CCAThresh << 8) | PowerComp);
  SPIDrvWrite(FEGAIN_ADDR, u16Reg);

  return SMAC_SUCCESS;
}
